<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class TherapistScheduleSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get all therapists
        $therapists = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'counselor')
            ->pluck('users.id')
            ->toArray();

        if (empty($therapists)) {
            $this->command->warn('No therapists found. Run TherapistSeeder first.');
            return;
        }

        // Get all appointments
        $appointments = DB::table('appointments')->get();

        $scheduleCount = 0;

        // Create schedules for each appointment
        foreach ($appointments as $appointment) {
            if (!$appointment->counselor_id) {
                continue;
            }

            $appointmentDate = Carbon::parse($appointment->preferred_date);
            
            // Create a schedule entry for this appointment
            DB::table('therapist_schedules')->insert([
                'therapist_id' => $appointment->counselor_id,
                'date' => $appointmentDate->toDateString(),
                'start_time' => $appointmentDate->format('H:i:s'),
                'end_time' => $appointmentDate->copy()->addHour()->format('H:i:s'),
                'status' => 'booked',
                'appointment_id' => $appointment->id,
                'notes' => null,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $scheduleCount++;
        }

        // Create additional available time slots for each therapist
        $startDate = Carbon::today();
        $endDate = Carbon::today()->addDays(14); // Next 2 weeks

        foreach ($therapists as $therapistId) {
            $currentDate = $startDate->copy();

            while ($currentDate <= $endDate) {
                // Skip Sundays
                if ($currentDate->dayOfWeek === Carbon::SUNDAY) {
                    $currentDate->addDay();
                    continue;
                }

                // Create available slots from 9 AM to 5 PM (excluding lunch 12-1)
                $timeSlots = [
                    ['09:00:00', '10:00:00'],
                    ['10:00:00', '11:00:00'],
                    ['11:00:00', '12:00:00'],
                    ['13:00:00', '14:00:00'],
                    ['14:00:00', '15:00:00'],
                    ['15:00:00', '16:00:00'],
                    ['16:00:00', '17:00:00'],
                ];

                foreach ($timeSlots as $slot) {
                    // Check if this slot is already booked
                    $existingSchedule = DB::table('therapist_schedules')
                        ->where('therapist_id', $therapistId)
                        ->where('date', $currentDate->toDateString())
                        ->where('start_time', $slot[0])
                        ->exists();

                    if (!$existingSchedule) {
                        // Create available slot
                        DB::table('therapist_schedules')->insert([
                            'therapist_id' => $therapistId,
                            'date' => $currentDate->toDateString(),
                            'start_time' => $slot[0],
                            'end_time' => $slot[1],
                            'status' => 'available',
                            'appointment_id' => null,
                            'notes' => null,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);

                        $scheduleCount++;
                    }
                }

                $currentDate->addDay();
            }
        }

        $this->command->info("Created {$scheduleCount} therapist schedule entries");
        $this->command->info("- Booked slots linked to appointments");
        $this->command->info("- Available slots for next 2 weeks");
    }
}
