<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportsTableSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Seed blog_posts
        $this->seedBlogPosts();

        // Seed case_categories
        $this->seedCaseCategories();

        // Seed analytics_events
        $this->seedAnalyticsEvents();

        // Seed client_surveys
        $this->seedClientSurveys();

        // Seed resources
        $this->seedResources();

        // Seed resource_bookings
        $this->seedResourceBookings();

        // Seed activity_logs
        $this->seedActivityLogs();

        // Seed seo_keywords
        $this->seedSeoKeywords();

        // Seed service_costs
        $this->seedServiceCosts();
    }

    private function seedBlogPosts(): void
    {
        // Skip if blog posts already exist
        if (DB::table('blog_posts')->count() > 0) {
            return;
        }

        $posts = [
            ['title' => 'Understanding Anxiety in Daily Life', 'views' => 1250, 'comments' => 45, 'shares' => 120],
            ['title' => 'Depression: Signs and Treatment Options', 'views' => 980, 'comments' => 32, 'shares' => 95],
            ['title' => 'Building Healthy Relationships', 'views' => 750, 'comments' => 28, 'shares' => 80],
            ['title' => 'Stress Management Techniques', 'views' => 1100, 'comments' => 38, 'shares' => 105],
            ['title' => 'Mental Health in the Workplace', 'views' => 650, 'comments' => 22, 'shares' => 60],
            ['title' => 'Sleep and Mental Health Connection', 'views' => 890, 'comments' => 35, 'shares' => 85],
            ['title' => 'Mindfulness for Beginners', 'views' => 1450, 'comments' => 52, 'shares' => 140],
            ['title' => 'Grief and Loss: Coping Strategies', 'views' => 720, 'comments' => 25, 'shares' => 70],
            ['title' => 'Self-Esteem and Confidence Building', 'views' => 960, 'comments' => 40, 'shares' => 92],
            ['title' => 'Family Therapy: When and Why', 'views' => 580, 'comments' => 18, 'shares' => 45],
        ];

        // Get first available user ID or use NULL
        $authorId = DB::table('users')->first()?->id;

        foreach ($posts as $post) {
            DB::table('blog_posts')->insert([
                'title' => $post['title'],
                'slug' => str_replace(' ', '-', strtolower($post['title'])),
                'content' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.',
                'views' => $post['views'],
                'comments' => $post['comments'],
                'shares' => $post['shares'],
                'author_id' => $authorId,
                'category' => ['Mental Health', 'Therapy', 'Wellness', 'Tips'][rand(0, 3)],
                'status' => 'published',
                'created_at' => Carbon::now()->subDays(rand(1, 90)),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedCaseCategories(): void
    {
        // Skip if case categories already exist
        if (DB::table('case_categories')->count() > 0) {
            return;
        }

        $categories = [
            ['name' => 'Anxiety Disorders', 'case_count' => 45, 'trend' => 5],
            ['name' => 'Depression', 'case_count' => 38, 'trend' => 2],
            ['name' => 'Relationship Issues', 'case_count' => 42, 'trend' => 8],
            ['name' => 'Work Stress', 'case_count' => 35, 'trend' => -3],
            ['name' => 'Grief & Loss', 'case_count' => 22, 'trend' => 1],
            ['name' => 'Life Transitions', 'case_count' => 28, 'trend' => 6],
            ['name' => 'Family Conflicts', 'case_count' => 31, 'trend' => 3],
            ['name' => 'Self-Esteem Issues', 'case_count' => 25, 'trend' => 4],
        ];

        foreach ($categories as $category) {
            DB::table('case_categories')->insert([
                'name' => $category['name'],
                'case_count' => $category['case_count'],
                'monthly_average' => round($category['case_count'] / 3),
                'trend_percentage' => $category['trend'],
                'season' => ['Spring', 'Summer', 'Fall', 'Winter'][rand(0, 3)],
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedAnalyticsEvents(): void
    {
        // Skip if analytics events already exist
        if (DB::table('analytics_events')->count() > 0) {
            return;
        }

        $eventTypes = ['page_view', 'button_click', 'form_submit', 'session_start'];
        $devices = ['Mobile', 'Desktop', 'Tablet'];
        $browsers = ['Chrome', 'Firefox', 'Safari', 'Edge'];
        $countries = ['Kenya', 'Uganda', 'Tanzania', 'Rwanda', 'Burundi'];
        $pages = ['/', '/about', '/services', '/contact', '/blog', '/appointment'];

        for ($i = 0; $i < 500; $i++) {
            DB::table('analytics_events')->insert([
                'event_type' => $eventTypes[array_rand($eventTypes)],
                'page_url' => $pages[array_rand($pages)],
                'referrer' => rand(0, 1) ? 'google.com' : 'direct',
                'device' => $devices[array_rand($devices)],
                'browser' => $browsers[array_rand($browsers)],
                'country' => $countries[array_rand($countries)],
                'occurred_at' => Carbon::now()->subDays(rand(1, 90))->subHours(rand(0, 23)),
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedClientSurveys(): void
    {
        // Skip if client surveys already exist
        if (DB::table('client_surveys')->count() > 0) {
            return;
        }

        $outcomes = ['improved', 'stable', 'needs_follow_up'];
        $satisfactionScores = [3.5, 4.0, 4.2, 4.5, 4.8, 5.0, 3.8, 4.3, 4.1, 4.7];
        $feedbacks = [
            'Very helpful session, gained new perspectives',
            'Counselor was understanding and supportive',
            'Practical advice I can apply in my life',
            'Good progress on my goals',
            'Sessions are very beneficial for me',
            'Feeling much better than before',
            'Counselor is professional and caring',
            'Helpful tools and strategies learned',
            'Positive change in my mental health',
            'Excellent service, highly recommend',
        ];

        // Get valid IDs from database
        $appointmentIds = DB::table('appointments')->pluck('id')->toArray() ?? [null];
        $userIds = DB::table('users')->pluck('id')->toArray() ?? [null];

        for ($i = 0; $i < 80; $i++) {
            $appointmentId = count($appointmentIds) > 0 ? $appointmentIds[array_rand($appointmentIds)] : null;
            $memberId = count($userIds) > 0 ? $userIds[array_rand($userIds)] : null;

            DB::table('client_surveys')->insert([
                'appointment_id' => $appointmentId,
                'member_id' => $memberId,
                'satisfaction_score' => $satisfactionScores[array_rand($satisfactionScores)],
                'feedback' => $feedbacks[array_rand($feedbacks)],
                'outcome_status' => $outcomes[array_rand($outcomes)],
                'progress_percentage' => rand(20, 100),
                'would_recommend' => rand(0, 100) > 15 ? 1 : 0,
                'completed_at' => Carbon::now()->subDays(rand(1, 60)),
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedResources(): void
    {
        // Skip if resources already exist
        if (DB::table('resources')->count() > 0) {
            return;
        }

        $resources = [
            ['name' => 'Therapy Room A', 'type' => 'Room', 'capacity' => 1],
            ['name' => 'Therapy Room B', 'type' => 'Room', 'capacity' => 1],
            ['name' => 'Group Session Room', 'type' => 'Room', 'capacity' => 10],
            ['name' => 'Meditation Space', 'type' => 'Facility', 'capacity' => 5],
            ['name' => 'Assessment Equipment', 'type' => 'Equipment', 'capacity' => 1],
            ['name' => 'Video Conferencing Setup', 'type' => 'Technology', 'capacity' => 1],
        ];

        foreach ($resources as $resource) {
            DB::table('resources')->insert([
                'name' => $resource['name'],
                'type' => $resource['type'],
                'capacity' => $resource['capacity'],
                'bookings_this_month' => rand(15, 50),
                'utilization_rate' => rand(60, 95),
                'status' => 'available',
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedResourceBookings(): void
    {
        // Skip if resource bookings already exist
        if (DB::table('resource_bookings')->count() > 0) {
            return;
        }

        // Get valid IDs from database
        $resourceIds = DB::table('resources')->pluck('id')->toArray() ?? [null];
        $appointmentIds = DB::table('appointments')->pluck('id')->toArray() ?? [null];

        for ($i = 0; $i < 150; $i++) {
            $resourceId = count($resourceIds) > 0 ? $resourceIds[array_rand($resourceIds)] : null;
            $appointmentId = count($appointmentIds) > 0 ? $appointmentIds[array_rand($appointmentIds)] : null;

            $booked_from = Carbon::now()->subDays(rand(1, 30))->setTime(rand(8, 16), 0);
            $booked_to = $booked_from->copy()->addHour();

            DB::table('resource_bookings')->insert([
                'resource_id' => $resourceId,
                'appointment_id' => $appointmentId,
                'booked_from' => $booked_from,
                'booked_to' => $booked_to,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedActivityLogs(): void
    {
        // Skip if activity logs already exist
        if (DB::table('activity_logs')->count() > 0) {
            return;
        }

        $actions = ['login', 'create', 'update', 'delete', 'view', 'export'];
        $modelTypes = ['Appointment', 'Invoice', 'User', 'Report', 'Booking'];

        // Get valid user IDs from database
        $userIds = DB::table('users')->pluck('id')->toArray() ?? [null];

        for ($i = 0; $i < 500; $i++) {
            $userId = count($userIds) > 0 ? $userIds[array_rand($userIds)] : null;

            DB::table('activity_logs')->insert([
                'user_id' => $userId,
                'action' => $actions[array_rand($actions)],
                'model_type' => $modelTypes[array_rand($modelTypes)],
                'model_id' => rand(1, 200),
                'description' => 'User performed ' . $actions[array_rand($actions)] . ' action',
                'changes' => json_encode(['field' => 'value']),
                'ip_address' => rand(192, 255) . '.' . rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255),
                'user_agent' => 'Mozilla/5.0',
                'logged_at' => Carbon::now()->subDays(rand(1, 90))->subHours(rand(0, 23)),
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedSeoKeywords(): void
    {
        // Skip if SEO keywords already exist
        if (DB::table('seo_keywords')->count() > 0) {
            return;
        }

        $keywords = [
            ['keyword' => 'counseling services kenya', 'search_volume' => 1500, 'current_rank' => 5, 'target_rank' => 1],
            ['keyword' => 'online therapy nairobi', 'search_volume' => 1200, 'current_rank' => 8, 'target_rank' => 1],
            ['keyword' => 'mental health support', 'search_volume' => 2000, 'current_rank' => 12, 'target_rank' => 1],
            ['keyword' => 'anxiety counseling', 'search_volume' => 950, 'current_rank' => 3, 'target_rank' => 1],
            ['keyword' => 'depression treatment', 'search_volume' => 1800, 'current_rank' => 15, 'target_rank' => 1],
            ['keyword' => 'relationship counseling', 'search_volume' => 1100, 'current_rank' => 7, 'target_rank' => 1],
            ['keyword' => 'stress management', 'search_volume' => 1400, 'current_rank' => 10, 'target_rank' => 1],
            ['keyword' => 'life coaching', 'search_volume' => 850, 'current_rank' => 20, 'target_rank' => 1],
            ['keyword' => 'family therapy', 'search_volume' => 700, 'current_rank' => 6, 'target_rank' => 1],
            ['keyword' => 'grief counseling', 'search_volume' => 600, 'current_rank' => 4, 'target_rank' => 1],
        ];

        foreach ($keywords as $kw) {
            DB::table('seo_keywords')->insert([
                'keyword' => $kw['keyword'],
                'search_volume' => $kw['search_volume'],
                'current_rank' => $kw['current_rank'],
                'target_rank' => $kw['target_rank'],
                'status' => $kw['current_rank'] <= 5 ? 'optimized' : 'monitoring',
                'ranking_change' => rand(-3, 2),
                'last_checked_at' => Carbon::now()->subDays(rand(1, 5)),
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }

    private function seedServiceCosts(): void
    {
        // Skip if service costs already exist
        if (DB::table('service_costs')->count() > 0) {
            return;
        }

        $services = [
            ['service_type' => 'Individual Counseling Session', 'labor_cost' => 800, 'material_cost' => 100, 'overhead_cost' => 200],
            ['service_type' => 'Group Therapy Session', 'labor_cost' => 1500, 'material_cost' => 150, 'overhead_cost' => 350],
            ['service_type' => 'Initial Assessment', 'labor_cost' => 600, 'material_cost' => 80, 'overhead_cost' => 150],
            ['service_type' => 'Crisis Intervention', 'labor_cost' => 1200, 'material_cost' => 200, 'overhead_cost' => 300],
            ['service_type' => 'Family Therapy Session', 'labor_cost' => 1400, 'material_cost' => 120, 'overhead_cost' => 280],
            ['service_type' => 'Online Counseling Session', 'labor_cost' => 700, 'material_cost' => 50, 'overhead_cost' => 150],
            ['service_type' => 'Couples Counseling', 'labor_cost' => 1100, 'material_cost' => 100, 'overhead_cost' => 250],
            ['service_type' => 'Career Counseling', 'labor_cost' => 900, 'material_cost' => 120, 'overhead_cost' => 180],
        ];

        foreach ($services as $service) {
            $total_cost = $service['labor_cost'] + $service['material_cost'] + $service['overhead_cost'];
            $selling_price = $total_cost * 1.5; // 50% markup
            $profit_margin = (($selling_price - $total_cost) / $selling_price) * 100;

            DB::table('service_costs')->insert([
                'service_type' => $service['service_type'],
                'labor_cost' => $service['labor_cost'],
                'material_cost' => $service['material_cost'],
                'overhead_cost' => $service['overhead_cost'],
                'total_cost' => $total_cost,
                'selling_price' => $selling_price,
                'profit_margin' => $profit_margin,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }
    }
}
