<?php

namespace Database\Seeders;

use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;

class DatabaseSeeder extends Seeder
{
    use WithoutModelEvents;

    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        $this->command->info('🌱 Starting database seeding...');

        // 1. Seed roles and permissions first (required for user assignments)
        $this->command->info('📋 Seeding roles and permissions...');
        $this->call([
            RoleSeeder::class,
        ]);

        // 2. Create or get admin user
        $this->command->info('👤 Creating admin user...');
        $admin = User::firstOrCreate(
            ['email' => 'admin@maakaba.com'],
            [
                'name' => 'Admin User',
                'password' => bcrypt('password'),
                'email_verified_at' => now(),
            ]
        );

        // Assign admin role if not already assigned
        if (!$admin->hasRole('admin')) {
            $admin->assignRole('admin');
        }

        // 3. Seed core site configuration
        $this->command->info('⚙️ Seeding site settings...');
        $this->call([
            SiteSettingsSeeder::class,
        ]);

        // 4. Seed content and services (no dependencies)
        $this->command->info('🏢 Seeding services and content...');
        $this->call([
            ServiceSeeder::class,
            TeamMemberSeeder::class,
            PostSeeder::class,
            ResourceSeeder::class,
        ]);

        // 5. Seed page content
        $this->command->info('📄 Seeding page content...');
        $this->call([
            HomePageSeeder::class,
            AboutPageSeeder::class,
            ContactPageSeeder::class,
            ServicesPageSeeder::class,
            TeamPageSeeder::class,
            MembershipPageSeeder::class,
            TestimonialsSeeder::class,
            OurApproachPageSeeder::class,
            FaqSeeder::class,
        ]);

        // 6. Seed membership system (depends on services)
        $this->command->info('💳 Seeding membership system...');
        $this->call([
            MembershipSeeder::class,
            MembershipComparisonSeeder::class,
            MemberSeeder::class,
            SubscriptionsSeeder::class,
        ]);

        // 7. Seed therapist system (required for appointments)
        $this->command->info('👩‍⚕️ Seeding therapist system...');
        $this->call([
            TherapistSeeder::class,
            TherapistScheduleSeeder::class,
        ]);

        // 8. Seed appointments (depends on therapists and members)
        $this->command->info('📅 Seeding appointments...');
        $this->call([
            AppointmentSeeder::class,
            CompleteBookingSystemSeeder::class,
        ]);

        // 9. Seed user-generated content
        $this->command->info('💬 Seeding user content...');
        $this->call([
            TestimonialSeeder::class,
            NotificationSeeder::class,
        ]);

        // 10. Seed activity logs and reports (depends on all other data)
        $this->command->info('📊 Seeding reports and activity logs...');
        $this->call([
            ActivityLogSeeder::class,
            ReportsTableSeeder::class,
        ]);

        $this->command->info('✅ Database seeded successfully with complete system data!');
        $this->command->info('🎯 Ready for development and testing!');
    }
}
